import InputLabel from "@/Components/InputLabel";
import InputError from "@/Components/InputError";
import TextInput from "@/Components/TextInput";
import Textarea from "@/Components/Textarea";
import PrimaryButton from "@/Components/PrimaryButton";
import __ from "@/Functions/Translate";
import { toast } from "react-toastify";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import { usePage, useForm, Head, Link } from "@inertiajs/inertia-react";
import { Inertia } from "@inertiajs/inertia";
import { useEffect, useState } from "react";
import axios from "axios";
import Spinner from "@/Components/Spinner";
import { MdVideoLibrary } from "react-icons/md";
import AccountNavi from "@/Pages/Channel/Partials/AccountNavi";
import { FaCloudUploadAlt, FaPhotoVideo } from "react-icons/fa";
import ImageUploading from "react-images-uploading";
import SecondaryButton from "@/Components/SecondaryButton";
import { RiDeleteBin5Line } from "react-icons/ri";
import { CiSaveUp1 } from "react-icons/ci";
import Modal from "@/Components/Modal";
import DangerButton from "@/Components/DangerButton";

export default function CreateGallery({ gallery }) {
  const { data, setData, post, processing, errors, progress } = useForm({
    gallery_name: gallery.gallery_name,
    preview_photo: gallery.preview_photo,
    price: gallery.price,
    free_for_subs: gallery.free_for_subs,
  });

  const onImagesChange = (imageList, addUpdateIndex) => {
    console.log(imageList);
    setImages(imageList);
  };

  const [images, setImages] = useState([]);
  const [isUploading, setIsUploading] = useState(false);
  const [showDeleteConfirmation, setShowDeleteConfirmation] = useState(false);
  const [deleteId, setDeleteId] = useState(0);
  const maxNumber = 16;

  const onHandleChange = (event) => {
    setData(
      event.target.name,
      event.target.type === "checkbox"
        ? event.target.checked
        : event.target.value
    );
  };

  const submit = (e) => {
    e.preventDefault();

    if (gallery.id) {
      post(route("gallery.update", { gallery: gallery.id }));
    } else {
      post(route("gallery.save"));
    }
  };

  useEffect(() => {
    if (Object.keys(errors).length) {
      Object.keys(errors).map((key) => {
        toast.error(errors.key);
      });
    }
  }, [errors]);

  const uploadImages = async (e) => {
    e.preventDefault();

    setIsUploading(true);

    if (images.length < 1) {
      toast.error(__("No images to upload"));
    }

    const formData = new FormData();

    images.forEach((img) => {
      console.log(img.file);

      formData.append("photos[]", img.file);
    });

    try {
      const response = await axios.post(
        route("gallery.uploadPhotos", { gallery: gallery.id }),
        formData
      );

      // You can add more logic here if necessary
      toast.success(__("Images uploaded successfully"));
      setIsUploading(false);
      setImages([]);
      Inertia.reload({ only: gallery });
    } catch (error) {
      setIsUploading(false);
      console.error("Error uploading images:", error);

      if (error.response) {
        // The request was made and the server responded with a status code
        // that falls out of the range of 2xx
        if (error.response.status === 422) {
          // Laravel validation error
          const errors = error.response.data.errors;

          // Display each error message using toast.error()
          Object.values(errors).forEach((errorMessages) => {
            errorMessages.forEach((errorMessage) => {
              toast.error(errorMessage);
            });
          });
        } else {
          toast.error(`Server Error: ${error.response.status}`);
        }
      } else if (error.request) {
        // The request was made but no response was received
        toast.error("No response from the server");
      } else {
        // Something happened in setting up the request that triggered an Error
        toast.error(`Error: ${error.message}`);
      }
    }
  };

  const confirmImageDelete = (e, id) => {
    e.preventDefault();

    setShowDeleteConfirmation(true);
    setDeleteId(id);
  };

  const deleteImageFromGallery = () => {
    Inertia.visit(route("gallery.photos.delete"), {
      method: "POST",
      data: { photo: deleteId },
      preserveState: false,
    });
  };

  return (
    <AuthenticatedLayout>
      <Head title={__("Create Gallery")} />

      <Modal
        show={showDeleteConfirmation}
        onClose={(e) => setShowDeleteConfirmation(false)}
      >
        <div className="px-5 py-10 text-center">
          <h3 className="text-xl mb-3 text-zinc-700 dark:text-white">
            {__("Are you sure you want to remove this photo?")}
          </h3>
          <DangerButton onClick={(e) => deleteImageFromGallery()}>
            {__("Yes")}
          </DangerButton>
          <SecondaryButton
            className="ml-3"
            onClick={(e) => setShowDeleteConfirmation(false)}
          >
            {__("No")}
          </SecondaryButton>
        </div>
      </Modal>

      <div className="lg:flex lg:space-x-10">
        <AccountNavi active={"gallery-manager"} />
        <div className="p-4 sm:p-8 bg-white w-full dark:bg-zinc-900 shadow sm:rounded-lg">
          <header className="mb-5">
            <h2 className="text-lg inline-flex items-center md:text-xl font-medium text-gray-600 dark:text-gray-100">
              <MdVideoLibrary className="mr-2" />
              {gallery.id === null ? __("Create Gallery") : __("Edit Gallery")}
            </h2>

            <p>
              <PrimaryButton
                onClick={(e) => Inertia.visit(route("gallery.manager"))}
              >
                {__("<< Back to Gallery Manager")}
              </PrimaryButton>
            </p>
          </header>
          <hr className="my-5" />
          <form onSubmit={submit} encType="multipart/form-data">
            <div className="mb-5">
              <InputLabel for="gallery_name" value={__("Gallery Name")} />

              <TextInput
                name="gallery_name"
                value={data.gallery_name}
                handleChange={onHandleChange}
                required
                className="mt-1 block w-full md:w-1/2"
              />

              <InputError message={errors.gallery_name} className="mt-2" />
            </div>

            <div className="flex w-full md:w-1/2 flex-col md:flex-row md:items-center md:space-x-10 md:justify-between">
              <div className="mb-5">
                <InputLabel for="price" value={__("Price")} />

                <div className="flex items-center">
                  <TextInput
                    type="number"
                    name="price"
                    value={data.price}
                    handleChange={onHandleChange}
                    required
                    className="mt-1  w-32"
                  />
                  <div className="ml-1 dark:text-white text-gray-700">
                    {__("tokens")}
                  </div>
                </div>

                <InputError message={errors.price} className="mt-2" />
              </div>
              <div className="mb-5">
                <InputLabel
                  for="free_for_subs"
                  value={__("Free for subscribers?")}
                />

                <select
                  name="free_for_subs"
                  value={data.free_for_subs}
                  onChange={onHandleChange}
                  required
                  className={`mt-1 block w-32 border-zinc-300 dark:border-zinc-700 dark:bg-zinc-800 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm `}
                >
                  <option value="yes">{__("Yes")}</option>
                  <option value="no">{__("No")}</option>
                </select>

                <InputError message={errors.free_for_subs} className="mt-2" />
              </div>
            </div>

            <div className="mb-5">
              <InputLabel
                for="preview_photo"
                value={__(
                  "Main Photo - helps to attract sales (will be resized to 220x340px)"
                )}
              />

              <TextInput
                className="p-1 block w-full md:w-1/2 text-gray-900 border border-gray-300 rounded-lg cursor-pointer bg-gray-50 dark:text-gray-300 focus:outline-none dark:bg-zinc-800 dark:border-gray-600 dark:placeholder-zinc-900"
                type="file"
                name="preview_photo"
                accept="image/jpeg,image/jpg,image/png"
                handleChange={(e) =>
                  setData("preview_photo", e.target.files[0])
                }
                required={gallery.id === null}
              />

              <InputError message={errors.preview_photo} className="mt-2" />
            </div>

            <div className="flex justify-between items-center">
              <PrimaryButton processing={processing}>
                {gallery.id === null
                  ? __("Save Gallery")
                  : __("Update Gallery")}
              </PrimaryButton>
            </div>

            {progress && (
              <progress className="mt-5" value={progress.percentage} max="100">
                {progress.percentage}%
              </progress>
            )}
          </form>
          {gallery.id && (
            <>
              <hr className="my-5" />
              <header className="my-5">
                <h2 className="text-lg inline-flex items-center md:text-xl font-medium text-gray-600 dark:text-gray-100">
                  <FaPhotoVideo className="mr-2" />
                  {__("Gallery Photos")}
                </h2>
              </header>

              {!isUploading && (
                <ImageUploading
                  multiple
                  value={images}
                  onChange={onImagesChange}
                  maxNumber={maxNumber}
                  dataURLKey="data_url"
                >
                  {({
                    imageList,
                    onImageUpload,
                    onImageRemoveAll,
                    onImageUpdate,
                    onImageRemove,
                    isDragging,
                    dragProps,
                  }) => (
                    <div className="mt-2">
                      <SecondaryButton
                        style={isDragging ? { color: "red" } : undefined}
                        onClick={onImageUpload}
                        {...dragProps}
                      >
                        <FaPhotoVideo className="mr-2 w-6 h-6" />
                        {__("Add Images")}
                      </SecondaryButton>
                      &nbsp;
                      <div className="mt-2 flex items-center flex-wrap space-x-3">
                        {imageList.map((image, index) => (
                          <div key={index} className="mt-2">
                            <img
                              src={image["data_url"]}
                              alt=""
                              width="120"
                              className="rounded-lg"
                            />
                            <div className="image-item__btn-wrapper">
                              <button onClick={() => onImageRemove(index)}>
                                <RiDeleteBin5Line className="text-red-600" />
                              </button>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </ImageUploading>
              )}

              {!isUploading && (
                <PrimaryButton onClick={(e) => uploadImages(e)}>
                  <FaCloudUploadAlt className="mr-2 w-5 h-5" />
                  {__("Save & Upload")}
                </PrimaryButton>
              )}

              {isUploading && <Spinner />}

              <div className="mt-5 flex items-center flex-wrap">
                {gallery.photos.length > 0 &&
                  gallery.photos.map((photo) => (
                    <div
                      className="mr-5 mt-2 relative text-center"
                      key={`photo-${photo.id}`}
                    >
                      <img
                        src={`${photo.thumbUrl}`}
                        alt=""
                        className="rounded-lg w-[220px] h-[340px]"
                      />
                      <button onClick={(e) => confirmImageDelete(e, photo.id)}>
                        <RiDeleteBin5Line className="text-red-600 h-6 w-6" />
                      </button>
                    </div>
                  ))}
              </div>
            </>
          )}
        </div>
      </div>
    </AuthenticatedLayout>
  );
}
